'use strict';

/* --------------------------------------------------------------
 graduated_prices.js 2017-03-09
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Graduate Prices Modal Controller
 *
 * Handles the graduate prices modal functionality.
 */
gx.controllers.module('graduated_prices', ['modal', gx.source + '/libs/info_box'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Creates a graduated prices form row.
     *
     * @param {Object} graduatedPrice Contains the graduated price information.
     *
     * @return {jQuery} Returns the created row selector.
     */
    function _createRow(graduatedPrice) {
        var $row = $('<div />', {
            'class': 'row',
            'data-graduated-price': graduatedPrice
        });

        var $quantityFormGroup = $('<div />', {
            'class': 'form-group col-xs-4'
        });

        var $quantityLabel = $('<label />', {
            'class': 'control-label',
            'text': jse.core.lang.translate('NUMBER_OF_PIECES', 'admin_quick_edit')
        });

        var $quantityInput = $('<input />', {
            'type': 'text',
            'class': 'form-control quantity',
            'value': graduatedPrice.quantity
        });

        $quantityFormGroup.append($quantityLabel, $quantityInput);

        var $priceFormGroup = $('<div />', {
            'class': 'form-group col-xs-4'
        });

        var $priceLabel = $('<label />', {
            'class': 'control-label',
            'text': jse.core.lang.translate('PRICE', 'admin_quick_edit')
        });

        var $priceInputGroup = $('<div />', {
            'class': 'input-group'
        });

        var $priceInput = $('<input />', {
            'type': 'text',
            'class': 'form-control price',
            'value': graduatedPrice.personal_offer
        });

        var $priceInputGroupButton = $('<span />', {
            'class': 'input-group-btn'
        });

        var $deleteButton = $('<button />', {
            'class': 'btn delete',
            'html': '<i class="fa fa-trash"></i>'
        });

        if (graduatedPrice.price_id === '') {
            $deleteButton.prop('disabled', true);
        }

        $priceInputGroupButton.append($deleteButton);

        $priceInputGroup.append($priceInput, $priceInputGroupButton);

        $priceFormGroup.append($priceLabel, $priceInputGroup);

        $row.append($quantityFormGroup, $priceFormGroup);

        return $row;
    }

    /**
     * Handles AJAX request errors.
     *
     * @param {jQuery.jqXHR} jqXHR jQuery request object.
     * @param {String} textStatus Request status string.
     * @param {Error} errorThrown Thrown error object.
     */
    function _handleRequestError(jqXHR, textStatus, errorThrown) {
        jse.libs.modal.message({
            title: jse.core.lang.translate('error', 'messages'),
            content: jse.core.lang.translate('UNEXPECTED_REQUEST_ERROR', 'admin_quick_edit')
        });
    }

    /**
     * Get graduated prices for selected product.
     *
     * @param {Number} productId Selected product ID.
     *
     * @return {jQuery.jqXHR} Returns request's deferred object.
     */
    function _getGraduatedPrices(productId) {
        return $.ajax({
            method: 'GET',
            url: jse.core.config.get('appUrl') + '/admin/admin.php?do=QuickEditOverviewAjax/ProductGraduatedPrices',
            data: {
                productId: productId,
                pageToken: jse.core.config.get('pageToken')
            },
            dataType: 'json'
        });
    }

    /**
     * Renders graduated prices content.
     *
     * @param {Object} graduatedPrices Contains graduated prices info for selected product.
     */
    function _displayGraduatedPrices(graduatedPrices) {
        var $modalBody = $this.find('.modal-body');

        $modalBody.empty();

        var $tabList = $('<ul />', {
            'class': 'nav nav-tabs',
            'role': 'tablist'
        });

        var $tabContent = $('<div />', {
            'class': 'tab-content'
        });

        for (var customerStatusId in graduatedPrices.data[0].customers) {
            var customerStatus = graduatedPrices.data[0].customers[customerStatusId];

            var $tab = $('<li />');

            var graduatedPricesCount = customerStatus.graduations.length ? ' (' + customerStatus.graduations.length + ')' : '';

            var $link = $('<a />', {
                'href': '#customer-status-' + customerStatusId,
                'role': 'tab',
                'data-toggle': 'tab',
                'html': customerStatus.status_name + graduatedPricesCount
            });

            $tab.append($link);

            $tabList.append($tab);

            // Add new tab container in tab content.
            var $tabPane = $('<div />', {
                'role': 'tabpanel',
                'class': 'tab-pane fade',
                'id': 'customer-status-' + customerStatusId,
                'data-customer-status-id': customerStatusId
            });

            var _iteratorNormalCompletion = true;
            var _didIteratorError = false;
            var _iteratorError = undefined;

            try {
                for (var _iterator = customerStatus.graduations[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                    var graduation = _step.value;

                    graduation.customer_status_id = customerStatusId;
                    $tabPane.append(_createRow(graduation));
                }
            } catch (err) {
                _didIteratorError = true;
                _iteratorError = err;
            } finally {
                try {
                    if (!_iteratorNormalCompletion && _iterator.return) {
                        _iterator.return();
                    }
                } finally {
                    if (_didIteratorError) {
                        throw _iteratorError;
                    }
                }
            }

            $tabPane.append(_createRow({
                price_id: '',
                quantity: '',
                personal_offer: '',
                customer_status_id: customerStatusId
            }));

            $tabContent.append($tabPane);
        }

        $modalBody.append($tabList, $tabContent);

        // Show the first tab contents.
        $tabList.find('a:first').tab('show');
    }

    /**
     * Updates row count in tab link.
     */
    function _updateTabCounters() {
        $this.find('.tab-pane').each(function () {
            var $tabPane = $(this);
            var graduatedPricesCount = $tabPane.find('.row').length - 1;
            var $tabLink = $this.find('[href="#' + $tabPane.attr('id') + '"]');
            var countText = graduatedPricesCount > 0 ? '(' + graduatedPricesCount + ')' : '';

            if ($tabLink.text().search(/\(.*\)/) !== -1) {
                $tabLink.text($tabLink.text().replace(/\(.*\)/, countText));
            } else {
                $tabLink.text($tabLink.text() + countText);
            }
        });
    }

    /**
     * Row input key up event handler.
     */
    function _onRowInputKeyUp() {
        var $row = $(this).parents('.row:first');
        var $lastRow = $row.parents('.tab-pane').find('.row:last');

        if ($lastRow[0] === $row[0] && $row.find('input.quantity').val() !== '') {
            var $tabPane = $row.parents('.tab-pane:first');

            $row.find('.btn.delete').prop('disabled', false);

            $tabPane.append(_createRow({
                price_id: '',
                quantity: '',
                personal_offer: '',
                customer_status_id: $tabPane.data('customerStatusId')
            }));

            _updateTabCounters();
        }
    }

    /**
     * Row delete button click event handler.
     */
    function _onRowDeleteClick() {
        $(this).parents('.row:first').remove();
        _updateTabCounters();
    }

    /**
     * Graduated prices modal show event handler.
     *
     * Loads and displays graduated price info for the selected product.
     */
    function _onModalShow() {
        _getGraduatedPrices($this.data('productId')).done(_displayGraduatedPrices).fail(_handleRequestError);
    }

    /**
     * Saves graduated prices and closes the modal.
     */
    function _onSaveClick() {
        var customerStatuses = {};

        $this.find('.tab-pane').each(function (index, tabPane) {
            var $tabPane = $(tabPane);
            var customerStatusId = $tabPane.data('customerStatusId');

            customerStatuses[customerStatusId] = [];

            $tabPane.find('.row').each(function (index, row) {
                var $row = $(row);

                if ($row.is(':last-child')) {
                    return false;
                }

                customerStatuses[customerStatusId].push({
                    price_id: $row.data('price_id'),
                    quantity: $row.find('input.quantity').val(),
                    personal_offer: $row.find('input.price').val()
                });
            });

            // Add value for empty groups.
            if (!customerStatuses[customerStatusId].length) {
                customerStatuses[customerStatusId].push('empty');
            }
        });

        $.ajax({
            method: 'POST',
            url: jse.core.config.get('appUrl') + '/admin/admin.php?do=QuickEditOverviewAjax/SaveGraduatedPrices',
            data: {
                pageToken: jse.core.config.get('pageToken'),
                productId: $this.data('productId'),
                customerStatuses: customerStatuses
            },
            dataType: 'json'
        }).done(function (response) {
            $this.modal('hide');
            jse.libs.info_box.addSuccessMessage(jse.core.lang.translate('SUCCESS_PRODUCT_UPDATED', 'admin_quick_edit'));
        }).fail(_handleRequestError);
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        $this.on('click', '.btn.delete', _onRowDeleteClick).on('click', '.btn.save', _onSaveClick).on('keyup', 'input.form-control', _onRowInputKeyUp).on('show.bs.modal', _onModalShow);

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
